/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2005 by Myricom, Inc.  All rights reserved.                 *
 *************************************************************************/

#ifndef MX__LIB_TYPES_H
#define MX__LIB_TYPES_H

#include "mx_auto_config.h"
#  include "myriexpress.h"
#  include "mx_extensions.h"
#include "mx__thread.h" /* MX_EVENT_T, MX__MUTEX_T, MX__THREAD_T */
#include "mx_io.h" /* mx_reg_t, mx_get_copyblock_t */
#include "mcp_events.h" /* mcp_uevt_t */
#  include "mx_debug.h" /* mx_assert */
#  include "mx__shim.h" /* mx_calloc */
#include "mx__rdma_alloc.h" /* mx_rdma_pool_t */
#undef SLIST_ENTRY
#include "bsd/queue.h" /* TAILQ */
#include "mcp_requests.h" /* mcp_ureq_t */
#include "mx__ack.h"

/* when changing one of the two enums below, please update mx__debug_dump.c
   for meaningful diagnostics */

enum mx__request_type {
  /* See the bottom of the file for a quick description of the life of
   * these requests */
  MX__REQUEST_TYPE_SEND_TINY = 0,
  MX__REQUEST_TYPE_SEND_SMALL,
  MX__REQUEST_TYPE_SEND_MEDIUM,
  MX__REQUEST_TYPE_SEND_LARGE,
  MX__REQUEST_TYPE_RECV,
  MX__REQUEST_TYPE_RECV_LARGE,
  MX__REQUEST_TYPE_RECV_SELF,
  MX__REQUEST_TYPE_RECV_SHM,
  MX__REQUEST_TYPE_CONNECT,
  MX__REQUEST_TYPE_CONNECT_REPLY,
  MX__REQUEST_TYPE_LIBACK,
};

#if MX_ONE_SIDED
#define MX__ONESIDED_MASK ((uint64_t)7 << 61)
#define MX__ONESIDED_PUT ((uint64_t)1 << 61)
#define MX__ONESIDED_GET ((uint64_t)2 << 61)
#define MX__ONESIDED_GET_REPLY ((uint64_t)3 << 61)
#define MX__ONESIDED_PUT_FAKE ((uint64_t)4 << 61)
#endif

enum mx__request_state {
  MX__REQUEST_STATE_SEND_QUEUED = 1, /* all requests */
  MX__REQUEST_STATE_PENDING = 2, /* recv, unexp */
  MX__REQUEST_STATE_BUFFERED = 4, /* null/small/medium send */
  MX__REQUEST_STATE_COMPLETED = 8,  /* all */
  MX__REQUEST_STATE_DEAD = 16, /* null/small/medium send */
  MX__REQUEST_STATE_ACKED = 32, /* all sends= > lib has acked */
  MX__REQUEST_STATE_MCP = 64, /* all requests */
  MX__REQUEST_STATE_REPLIED = 128, /* large send got notified, or connect got replied */
  MX__REQUEST_STATE_RECV_MATCHED = 256, /* the recv request got matched */
  MX__REQUEST_STATE_INTERNAL = 512, /* one-side/--disable-rndv */
  MX__REQUEST_STATE_FREED = 16384, /* on free list */
  MX__REQUEST_STATE_DBGFLAG = 32768, /* for conservation of matter */
};

enum mx__wait_state {
  MX__WAIT_STATE_COMPLETE = 1,
  MX__WAIT_STATE_WAITING = 2,
  MX__WAIT_STATE_ABORTED = 3
};

struct mx__wait_queue_head {
  struct mx__wait_queue_head *prev, *next;
};

struct mx__wait_queue {
  enum mx__wait_state wait_state;
  MX__EVENT_T event;
  struct mx__wait_queue_head queue_elt;
  uint32_t result; /* todo: is this field redundant with wait_state ? */
  union {
    struct {
      uint64_t match_info;
      uint64_t match_mask;
      union mx_request * request;
      uint8_t is_wait_any;
    } peek;
    struct {
      uint64_t match_info;
      uint64_t match_mask;
      union mx_request * request;
    } probe;
  } type;
};

struct mx__request_queue_head {
  struct mx__request_queue_head *prev, *next;
};

/* do not use the same struct than above so that we get more type checking */
struct mx__partner_request_queue_head {
  struct mx__partner_request_queue_head *prev, *next;
};

struct mx__basic_request {
  struct mx__request_queue_head queue_elt;
  struct mx__partner_request_queue_head partner_queue_elt;
  enum mx__request_type type;
  uint32_t state;
  mx_status_t status;
  uint16_t mcp_handle;
  struct mx__wait_queue * wq;
  uint16_t send_seq;
  uint16_t  requeued; /* true if the lib is requeuing this request. Don't
		     use a new sequence number, use what the lib
		     provides. */
  mx_jiffies_t timeout;
  mx_jiffies_t last_send_time;
  mx_jiffies_t first_send_time;
  uint8_t acquired_by_wait_any;
  struct mx__partner *partner;
};

typedef struct {
  uint64_t data_target;
  uint32_t length;
  uint32_t pad;
} mx__get_t;

#define MX__MAX_TINY_LEN 32
#define MX__MAX_SMALL_LEN 128

union mx_request {
  struct {
    struct mx__basic_request basic;

    mx_segment_t *segments;
    uint32_t count;
    mx_segment_t segment;
    uintptr_t memory_context;
    uint32_t local_rdma_id;
    uint32_t offset_data;
    mx_reg_t reg_area;
    void *contiguous_copy;
    void *caller; /* debug: pointer to caller of mx_isend */
    uint64_t shm_magic;
    uint32_t frag_off;
    uint32_t accum;
    uint8_t peer_rdma;
    uint64_t shadow[MX__MAX_SMALL_LEN/sizeof(uint64_t)];
  } send;
  struct {
    struct mx__basic_request basic;

    mx_segment_t *segments;
    uint32_t count;
    mx_segment_t segment; /* if only one, which is the general case */
    uintptr_t memory_context;
    uint64_t match_info;
    uint64_t match_mask;

    uint32_t offset_data;
    uint32_t r_length;
    uint16_t msg_seq;
    uint32_t accum; /* amount of bytes received but not necessarily copied */
    uint32_t r_mask; /* bitmask of pieces to receive */
    uint8_t unexpected;
    uint8_t put_target;
    uint8_t self_send;
    union mx_request *self_send_req;
    mx_reg_t reg_area;
    void *contiguous_copy;
    uint32_t local_rdma_id;
    /* the rdma window, seqnum and offset are copied to/from lanai without
       endianess convertion, they should be considered opaque */
    uint32_t remote_rdma;
    uint64_t shm_peer_req;
    mx_shm_seg_t shm_src_segs;
    uint32_t shm_src_nsegs;
    uint32_t shm_src_session;
    uint16_t shm_peer_endpt;
    uint32_t ordered_unexp_weight;
    uint8_t notifying;
  } recv;
  struct {
    struct mx__basic_request basic;
    uint8_t is_synchronous;
    uint8_t peer_endpoint_id;
    uint16_t peer_index_n;
    uint16_t seqnum_start;
    uint8_t status_code_n;
    uint8_t connect_seqnum_n;
    uint32_t app_key_n;
    uint32_t dest_session_n;
  } connect;
  struct {
    struct mx__basic_request basic;
    uint16_t peer_n;
    uint8_t eid;
  } liback;
  struct mx__basic_request basic;
};

struct mx__request_lookaside
{
  struct mx__request_queue_head request_buffers_queue; /* SINGLY-LINKED list of request buffers */
  struct mx__request_queue_head free_requests_queue; /* SINGLY-LINKED list of free requests */
  int count;
  int alloc_count;
};

struct mx__connect {
  MX__THREAD_T thread;
  uint32_t key;
};

#if MX_ENABLE_STATS || MX_DEBUG
#define MX__EP_STATS 1
#else
#define MX__EP_STATS 0
#endif

#if MX__EP_STATS
struct mx__ep_stats
{
  /* send */
  int isend_tiny;
  int isend_small;
  int isend_medium;
  int isend_large;
  int issend;
  /* recv */
  int irecv;
  int expected;
  int unexpected;
  int unexpected_handled;
  int early;
  /* overlap */
  int test;
  int noncompleted_test;
  int wait;
  int nonblocking_wait;
  int completion;
  int overlapped_completion;

  /* ack */
  int resent;
  int resent_slow;
  int delayed_acks;
  int total_acks;

  /* rcache */
  int rcache_hit;
  int rcache_miss;
  unsigned long long rcache_hit_kbytes;
  unsigned long long rcache_miss_kbytes;
  /* misc */
  int no_mcp_handle;
  int got_mcp_handle;
};

#define MX__EP_STATS_INC(endpoint, field) endpoint->stats.field++
#define MX__EP_STATS_ADD(endpoint, field, val) \
      do { endpoint->stats.field += (val); } while (0)
#else /* MX__EP_STATS */
#define MX__EP_STATS_INC(endpoint, field)
#define MX__EP_STATS_ADD(endpoint, field, val)
#endif /* MX__EP_STATS */

struct mx_endpoint
{
  MX__MUTEX_T lock;
  MX__THREAD_T thread;
  int cancelled;
  struct mx_endpoint *next;

  mx_endpt_handle_t handle;
  mx_get_copyblock_t desc;
  char *sram;
  char *ze_req;
  mx_kernel_window_t *kernel_window;

  char *sendq;
  struct mx__memory_pool *send_pool;

  /* EVENT: */
  char *eventq;
  int eventq_length;
  mcp_uevt_t *eventq_uevt;
  uint32_t eventq_index;
  uint32_t eventq_flow;
  uint32_t event_count; /* number of events processed */

  /* RECV: */
  char *recvq;
  int recvq_length;
  int recvq_loc;

  struct mx__mcp_request_ring *req_ring;

  char *udataq;
  struct mx__ptr_stack *small_msg_ptrs;

  /* TODO: More fields may be added in the future in which case this
     becomes a struct. */
  uint64_t *flow;

  struct mx__request_lookaside req_lookaside;

  /* context ids */
  uint8_t ctxid_bits;
  uint32_t ctxid_max;
  uint8_t ctxid_shift;
  uint64_t ctxid_mask;

  /* context id structures */
  struct {
    /* Receive that have not been matched */
    struct mx__request_queue_head recv_reqq;
    /* All requests (except synchronous connect) when COMPLETED */
    struct mx__request_queue_head doneq;
    /* Unexpected receive, even incomplete mediums, before being matched
     * (they will be merged with the matching receive
     * and moved to the doneq or to the multifrag_recvq) */
    struct mx__request_queue_head unexpq;

    struct mx__wait_queue_head peek_queue_head;
    struct mx__wait_queue_head probe_queue_head;

    /* could move peek/probe_waiters here but they are required
     * at the endpoint level for mx_wakeup and mx__sleep_on_request
     * does not support two waiters counters for now */
  } * ctxid;

  /* those are basic queues (internal use only) */

  /* Requests that are queued for sending for the first time
   * (SEND_QUEUED with basic.queued = 0) */
  struct mx__request_queue_head send_reqq;
  /* Requests that are queued for sending again
   * (SEND_QUEUED with basic.queued > 0)
   * Libacks that are queued for sending (they won't be resent afterwards) */
  struct mx__request_queue_head resend_reqq;
  /* Tiny/small/medium send that have been buffered and posted to the MCP,
   * waiting for the MCP to return the done events */
  struct mx__request_queue_head buffered_sendq;
  /* Large send that have been posted to the MCP,
   * waiting for the MCP to return the done events */
  struct mx__request_queue_head large_sendq;
  /* Large send that have been completed by the MCP
   * (eventually not been acked), waiting for the notify to arrive */
  struct mx__request_queue_head notifying_large_sendq;
  /* Large receive that have been posted to the MCP,
   * waiting for the MCP to return the done event, 
   * they will send the notify then */
  struct mx__request_queue_head large_getq;
  /* Expected medium receive that have matched
   * and received at least one fragment */
  struct mx__request_queue_head multifrag_recvq;
  /* Connect requests that have posted to the MCP,
   * waiting for the MCP to return the done event */
  struct mx__request_queue_head mcp_connectq;
  /* Connect reply requests that have been posted to the MCP,
   * waiting for the MCP to return the done event, they will be dropped
   * afterwards since there is no need to retransmit them */
  struct mx__request_queue_head mcp_connect_replyq;
  /* Requests that have been sent and are waiting for a retransmit timeout
   * to raise, they will be moved to the resend_reqq then */
  struct mx__request_queue_head resend_list;
  /* Libacks that have been posted to the MCP,
   * waiting for the MCP to return the done event, they will be dropped
   * afterwards since there is no need to retransmit them */
  struct mx__request_queue_head ackq;

  struct mx__wait_queue_head wait_queue_head;

  int wait_waiters;
  int peek_waiters;
  int probe_waiters;

  mx_jiffies_t timeout;
  mx_jiffies_t resend_delay;
  mx_jiffies_t ack_delay;
  uint8_t timer;

  uint32_t sendself_count;
  uint32_t sendshm_count;
  /* connect requests are in no queue after being removed from the connectq
   * when the reply arrives */
  uint32_t connect_count;
  uint32_t liback_count;
  uint32_t acquired_by_wait_any_count;

  struct mx__handle_map *handle_map;
  uint32_t max_mcp_handles;

  mx_rdma_pool_t rdmas;
  struct mx_rdma_req {
    union mx_request *req;
    uint8_t seqno;
#if MX_OS_UDRV
    void *copy;
    uintptr_t copy_offset;
#endif
  } *rdma_requests;

  struct mx__partner * myself;
  uint32_t endpoint_sid_n;
  int board_num;
  int board_type;
  int is_ze;

  uint32_t tiny_msg_threshold;  /* threshold between one pio and two pio */
  uint32_t small_msg_threshold;  /* threshold between two pio and copy */
  uint32_t medium_msg_threshold; /* threshold between copy and dma */
  uint8_t medium_msg_pipelines[64];
  
  uint32_t max_endpoints;
  uint32_t max_peers;
  struct mx__partner **remote_ep;
  int32_t *application_rank_info; /* used for diagnostic only */
  struct mx__connect connect;
#if MX__EP_STATS
  struct mx__ep_stats stats;
#endif
  TAILQ_HEAD(s_rdmawin_pinned, mx__rdmawin) rdmawin_pinned;
  TAILQ_HEAD(s_rdmawin_free, mx__rdmawin) rdmawin_free;
  struct mx__rdmawin *rdmawin_items;
  uint32_t wake_pending;
  int wake_mcp_handle;

  mx_unexp_handler_t unexp_handler;
  void *unexp_handler_context;
  int in_handler;
  MX__EVENT_T in_handler_event;

  int app_waiting;					  
  int in_progression_thread;

  mx_error_handler_t error_handler;
  uint32_t unexp_queue_length;
  uint32_t ordered_unexp_length;
  uint32_t unexp_queue_length_max;
  uint64_t get_id;
#define MX__MAX_DEAD 512
  uint32_t dead_count;

  uint8_t lxgdb;

  struct mx__shm_info *shm;
  TAILQ_HEAD(s_partner, mx__partner) partners_to_ack;
};

typedef void (*mx__process_recv_msg_t)(mx_endpoint_t ep, union mx_request *r, 
				       mcp_uevt_msg_t *evt, void *data);

struct mx__early_queue_head {
  struct mx__early_queue_head *prev, *next;
};

struct mx__early
{
  struct mx__early_queue_head queue_elt;
  /* EVENT: */
  mcp_uevt_tiny_t recv_tiny;
  uint8_t type;
  mx__process_recv_msg_t recv_func;
  char *data;
  uint16_t data_length;
  uint16_t msg_seq;
};

#define MX__SEQNO_CNT 0x4000U

/* We need at least 2 queues so that seqnum rollover does generate any unordered queue */
#define MX__PARTNER_PENDING_SEQNUM_TO_QUEUE(partner,seqnum) ((partner)->pendingq)

struct mx__partner
{
  TAILQ_ENTRY(mx__partner) ack_list;
  mx_jiffies_t oldest_recv_time;
  mx_jiffies_t last_ack;
  uint16_t send_seq;
  uint16_t recv_seq;
  uint16_t fully_recv_seq; /* only incremented after all partials */
  uint16_t quadrant_count[4];

  /* Sends that have not been acked, ordered by seqnum */
  struct mx__partner_request_queue_head pendingq;

  /* Medium receive (queued by partner_queue_elt), either expected or not,
   * that have received at least one fragment but not all */
  struct mx__partner_request_queue_head partialq;

  /* Early fragments, either tiny/small/medium/rndv/notify,
   * before they are actually received */
  struct mx__early_queue_head early_queue;

  uint64_t nic_id;
  uint32_t endpoint_sid_n;
  uint32_t connect_session_n;
  uint32_t best_session_n;
  uint32_t app_key_n;
  uint16_t peer_index_n;
  uint8_t eid;
  int8_t connect_recvseq;
  int8_t connect_sendseq;
  union mx_request *liback_pending;
  uint16_t send_acked;
  uint16_t recv_acked;
  uint32_t recv_acknum;
  uint32_t send_acknum;
  uint8_t requeued;
  void *context;
};

#define MX__UREQ(ep, handle)									\
     ((ep)->is_ze ? 										\
         (ep)->req_ring->base + ((handle)*(64/sizeof(mcp_ureq_t))) % (256 / sizeof(mcp_ureq_t)):	\
      mx__get_ureq((ep)->req_ring,(handle)))
#endif




/***************************************************
 * Description of the life of a request.
 * (up-to-date as of september 2006)
 */

/*
  Send Tiny
  =========
  1) if enough resources, go to 4)
  2) mark SEND_QUEUED, enqueue in send_reqq
  3) mx__process_requests() removes SEND_QUEUED and dequeues
  4) mx__post_send() posts the request, marks BUFFERED and MCP and enqueues in buffered_sendq
  5) mx__process_events() dequeues from buffered_sendq, removes MCP, enqueues in resend_list
  6) mx__process_resend_list() adds SEND_QUEUED, dequeues from resend_list and queues in resend_reqq
  7) mx__process_requests removes() SEND_QUEUED and dequeues
  8) mx__post_send() posts the request, marks MCP and enqueues in buffered_sendq
  9) mx__process_events() dequeues from buffered_sendq, removes MCP, enqueues in resend_list
  10) goto 6) until ACKED
  11) when acked, mx__mark_request_acked() dequeues from resend_list and calls mx__send_acked_and_mcp_complete()
  12) mx__send_complete() queues in doneq/ctxid
  13) mx_test/wait/ipeek/peek/test_any/wait_any() dequeues from doneq/ctxid
*/

/*
  Send Small
  ==========
  As Tiny, except:
  4,8) mx__post_send() calls mx__ptr_stack_pop() to get room to write the data.
  5,9) mx__process_events() calls mx__ptr_stack_push() to release room.
*/

/*
  Send Medium
  ===========
  As Tiny, except:
  skip 1), do 2,3)
  4) mx_post_send() calls mx__memory_pool_alloc() to get room in the send copyblock
  11) mx__send_acked_and_mcp_complete() calls mx__release_send_medium() which calls mx__memory_pool_free()
*/

/*
  Send Large
  ==========
  1) mark SEND_QUEUED, enqueue in send_reqq
  2) mx__process_requests() removes SEND_QUEUED and dequeues
  3) mx__post_send() removes SEND_QUEUED, adds MCP, allocates rdma window, post the rdnv and enqueues to large_sendq
  4) mx__process_events() removes MCP, dequeues from large_sendq, enqueues to resend_list
  5) mx__process_resend_list() adds SEND_QUEUED, dequeues from resend_list and queues in resend_reqq
  6) mx__process_requests() removes SEND_QUEUED and dequeues
  7) mx__post_send() posts the request, marks MCP and enqueues in large_sendq
  8) mx__process_events() removes MCP, dequeues from large_sendq, enqueues to resend_list
  9) goto 5) until ACKED
  10) mx__send_acked_and_mcp_complete() enqueues notifying_large_sendq
  11) mx__process_receives() gets a MX_MCP_UEVT_RECV_NOTIFY, dequeues from notifying_large_sendq, calls mx__process_recv_notify() which calls mx__rndv_got_notify()
  12) mx__release_send_large() releases the rdma window
  12) mx__send_complete() queues in doneq/ctxid
  13) mx_test/wait/ipeek/peek/test_any/wait_any() dequeues from doneq/ctxid
*/

/*
  Incoming Message
  ================
  1) mx__process_events() calls mx__process_recvs() with the callback associated to the event type
  2) mx__process_recvs() checks whether the message is obsolete and drops it then
  2) mx__process_recvs() checks whether the message is early and stores it in the early queue then
  3) mx__process_ordered_evt() is called on the message if it is the expected seqnum
  4) mx__endpoint_match_receive() tries to match the message in the receive queue
  4.a) if unexpected (no match found), mx__create_unexp_for_evt() is called to generate an unexpected request and store it in the unexp queue
  4.b) if expected (match found), the corresponding receive requests is filled with the message info, and MX__REQUEST_STATE_RECV_MATCHED is set
  5) the callback is called
  5.a) for tiny messages, mx__process_recv_tiny() copies the data from the event and calls mx__recv_complete if the message is expected
  5.b) for small messages, mx__process_recv_tiny() copies the data from the copy block and calls mx__recv_complete if the message is expected
  5.c) for medium messages, mx__process_recv_medium() moves the requests in the multifrag_recvq if the message is expected, inserts in the partner partialq if it is not that last fragment, and calls mx__process_recv_copy_frag()
  5.c.1) mx__process_recv_copy_frag() copies the fragment from the copy block and calls mx__received_last_frag() if it was the last fragment
  5.c.2) mx__received_last_frag() removes the partner partialq if there's more than one fragment, removes from the multifrag_recvq if expected, and calls mx__recv_complete()
  5.d) for large message, mx__process_recv_large() calls mx__queue_large_recv() if the message is expected (see Recv Large below)
  6) mx__process_early() is called to process the next seqnum messages if already arrived
*/

/*
  Recv Posting
  ============
  1) mx_irecv() calls mx__endpoint_match_unexpected() to try to match the new receive in the unexp queue, if not, the recv is enqueued in the recv_reqq, else go on
  2) the receive request gets info from the unexpected requests, and MX__REQUEST_STATE_RECV_MATCHED is set
  3) if not large, the data is copied from the unexpected
  4.a) if medium, not entirely received, the unexpected is replaced with the receive request in the partner's partialq, and the receive request is enqueue in the multifrag_recvq
  4.b) if large, mx__queue_large_recv() is called (see Recv Large below)
  5) the unexpected requests is released
*/

/*
  Recv Large
  ==========
  1) mx__queue_large_recv() changes the type from RECV to RECV_LARGE, and sets SEND_QUEUED
  2) enqueue to send_rreq and if length>0 then mark notifying and goto 5)
  3) mx__process_requests() removes SEND_QUEUED, adds MCP, allocates rdma window, calls mx__post_large_recv(), and enqueues to large_getq
  4) mx__process_events() removes MCP, adds SEND_QUEUED, sets notifying, removes from large_getq and enqueues in send_rreq
  5) mx__process_requests() calls mx__post_send() and enqueues to large_getq
  6) mx__process_events() removes MCP, removes from large_getq and calls mx__send_acked_and_mcp_complete() if ACKED or go back to 4)
  7) mx__release_recv_large() releases rdma window
  8) mx__recv_complete() enqueues in doneq/ctxid
*/

/*
  Shmem
  =====
  TODO
*/

/*
  Self
  ====
  TODO
*/

/*
  Connect
  =======
  1) mx__connect_common() sets SEND_QUEUED and enqueues in REsend_reqq
  2) mx__process_requests() removes SEND_QUEUED, adds MCP, post the request and enqueues in mcp_connectq
  3) mx__process_events() dequeues from mcp_connectq, removes MCP, enqueues in resend_list
  4) mx__process_resend_list() adds SEND_QUEUED, dequeues from resend_list and queues in resend_reqq
  5) mx__process_requests() removes SEND_QUEUED, adds MCP, post the request and enqueues in mcp_connectq
  6) goto 3) until REPLIED
  7) mx__process_events() dequeues from mcp_connectq, removes MCP
  8) mx__connect_complete() enqueues in doneq/ctxid
  9) mx_test/wait/ipeek/peek/test_any/wait_any() dequeues from doneq/ctxid
*/

/*
  Connect Reply
  =============
  1) mx__process_events() gets a MX_MCP_UEVT_RECV_CONNECT and calls mx__handle_connect()
  2) mx__handle_connect() checks the connect request and queues a connect reply with SEND_QUEUED to REsend_reqq
  3) mx__process_requests() removes SEND_QUEUED, adds MCP, post the request and enqueues in connect_replyq
  4) mx__process_requests() removes SEND_QUEUED, dequeues from connect_replyq, releases the request and that's it (no need to ack)
*/

/*
  Lib Acks
  ========
  1) mx__queue_liback sets SEND_QUEUED and enqueues in REsend_reqq
  2) mx__process_requests() removes SEND_QUEUED, adds MCP, post the request and enqueues in ackq
  3) mx__process_events() removes from ackq and releases the request (no need to ack a ack :))
*/
